
  /**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, InspectorControls, MediaUpload, MediaUploadCheck } from '@wordpress/block-editor';

/**
 * WordPress components for the inspector panel
 */
import {
	PanelBody,
	SelectControl,
	TextControl,
	TextareaControl,
	ToggleControl,
	Button,
	__experimentalHeading as Heading,
	CardDivider
} from '@wordpress/components';

/**
 * React hooks
 */
import { useState } from '@wordpress/element';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @param {Object} props Block properties
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes }) {
	const { theme, cards } = attributes;
	const [selectedCardIndex, setSelectedCardIndex] = useState(0);

	const blockProps = useBlockProps({
		className: `wp-block-telex-bento-grid-5 theme-${theme}`,
	});

	const updateCard = (index, field, value) => {
		const updatedCards = [...cards];
		updatedCards[index] = { ...updatedCards[index], [field]: value };
		setAttributes({ cards: updatedCards });
	};

	const updateCardImage = (index, media) => {
		const updatedCards = [...cards];
		updatedCards[index] = { 
			...updatedCards[index], 
			image: { 
				url: media.url, 
				alt: media.alt || media.title || ''
			} 
		};
		setAttributes({ cards: updatedCards });
	};

	const removeCardImage = (index) => {
		const updatedCards = [...cards];
		updatedCards[index] = { 
			...updatedCards[index], 
			image: { url: '', alt: '' }
		};
		setAttributes({ cards: updatedCards });
	};

	const LinkIcon = () => (
		<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
			<path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71" />
			<path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71" />
		</svg>
	);

	return (
		<>
			<InspectorControls>
				<PanelBody title={__('Design Theme', 'bento-grid-5-block-wp')} initialOpen={true}>
					<SelectControl
						label={__('Choose Theme', 'bento-grid-5-block-wp')}
						value={theme}
						options={[
							{ label: __('Modern Light', 'bento-grid-5-block-wp'), value: 'modern-light' },
							{ label: __('Bold Dark', 'bento-grid-5-block-wp'), value: 'bold-dark' }
						]}
						onChange={(value) => setAttributes({ theme: value })}
					/>
				</PanelBody>

				<PanelBody title={__('Card Settings', 'bento-grid-5-block-wp')} initialOpen={true}>
					<SelectControl
						label={__('Select Card to Edit', 'bento-grid-5-block-wp')}
						value={selectedCardIndex}
						options={cards.map((card, index) => ({
							label: __(`Card ${index + 1}: ${card.title}`, 'bento-grid-5-block-wp'),
							value: index
						}))}
						onChange={(value) => setSelectedCardIndex(parseInt(value))}
					/>

					<CardDivider />

					{cards[selectedCardIndex] && (
						<>
							<Heading level={4}>{__(`Editing Card ${selectedCardIndex + 1}`, 'bento-grid-5-block-wp')}</Heading>
							
							<MediaUploadCheck>
								<MediaUpload
									onSelect={(media) => updateCardImage(selectedCardIndex, media)}
									allowedTypes={['image']}
									value={cards[selectedCardIndex].image.url}
									render={({ open }) => (
										<div style={{ marginBottom: '16px' }}>
											<Button 
												onClick={open} 
												variant="secondary" 
												style={{ marginBottom: '8px', display: 'block', width: '100%' }}
											>
												{cards[selectedCardIndex].image.url ? 
													__('Change Image', 'bento-grid-5-block-wp') : 
													__('Upload Image', 'bento-grid-5-block-wp')
												}
											</Button>
											{cards[selectedCardIndex].image.url && (
												<>
													<img 
														src={cards[selectedCardIndex].image.url} 
														alt={cards[selectedCardIndex].image.alt}
														style={{ width: '100%', height: 'auto', maxHeight: '120px', objectFit: 'cover', marginBottom: '8px', borderRadius: '4px' }}
													/>
													<Button 
														onClick={() => removeCardImage(selectedCardIndex)} 
														variant="link" 
														isDestructive
													>
														{__('Remove Image', 'bento-grid-5-block-wp')}
													</Button>
												</>
											)}
										</div>
									)}
								/>
							</MediaUploadCheck>

							<TextControl
								label={__('Image Alt Text', 'bento-grid-5-block-wp')}
								value={cards[selectedCardIndex].image.alt}
								onChange={(value) => {
									const updatedCards = [...cards];
									updatedCards[selectedCardIndex] = { 
										...updatedCards[selectedCardIndex], 
										image: { 
											...updatedCards[selectedCardIndex].image,
											alt: value
										}
									};
									setAttributes({ cards: updatedCards });
								}}
							/>

							<TextControl
								label={__('Title', 'bento-grid-5-block-wp')}
								value={cards[selectedCardIndex].title}
								onChange={(value) => updateCard(selectedCardIndex, 'title', value)}
							/>

							<TextareaControl
								label={__('Description', 'bento-grid-5-block-wp')}
								value={cards[selectedCardIndex].description}
								onChange={(value) => updateCard(selectedCardIndex, 'description', value)}
							/>

							<TextControl
								label={__('Link URL', 'bento-grid-5-block-wp')}
								value={cards[selectedCardIndex].link}
								onChange={(value) => updateCard(selectedCardIndex, 'link', value)}
								help={__('Leave empty to disable the link', 'bento-grid-5-block-wp')}
							/>

							<ToggleControl
								label={__('Show Description', 'bento-grid-5-block-wp')}
								checked={cards[selectedCardIndex].showDescription}
								onChange={(value) => updateCard(selectedCardIndex, 'showDescription', value)}
							/>

							<ToggleControl
								label={__('Show Link Icon', 'bento-grid-5-block-wp')}
								checked={cards[selectedCardIndex].showIcon}
								onChange={(value) => updateCard(selectedCardIndex, 'showIcon', value)}
							/>

							<TextControl
								label={__('Custom CSS Class', 'bento-grid-5-block-wp')}
								value={cards[selectedCardIndex].customClass}
								onChange={(value) => updateCard(selectedCardIndex, 'customClass', value)}
								help={__('Add custom CSS classes for styling', 'bento-grid-5-block-wp')}
							/>
						</>
					)}
				</PanelBody>
			</InspectorControls>

			<div {...blockProps}>
				<ul className="bento-grid">
					{cards.map((card, index) => (
						<li 
							key={card.id} 
							className={`bento-card ${card.customClass ? card.customClass : ''}`}
							onClick={() => setSelectedCardIndex(index)}
							style={{ cursor: 'pointer' }}
						>
							{card.image.url && (
								<figure className="card-image">
									<img 
										src={card.image.url} 
										alt={card.image.alt}
										loading="lazy"
									/>
								</figure>
							)}
							<div className="card-content">
								<h3 className="card-title">{card.title}</h3>
								{card.showDescription && card.description && (
									<p className="card-description">{card.description}</p>
								)}
								{card.showIcon && card.link && (
									<div className="card-link-icon">
										<LinkIcon />
									</div>
								)}
							</div>
						</li>
					))}
				</ul>
			</div>
		</>
	);
}
	