
  /**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps } from '@wordpress/block-editor';

/**
 * The save function defines the way in which the different attributes should
 * be combined into the final markup, which is then serialized by the block
 * editor into `post_content`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#save
 *
 * @param {Object} props Block properties
 * @return {Element} Element to render.
 */
export default function save({ attributes }) {
	const { theme, cards } = attributes;

	const blockProps = useBlockProps.save({
		className: `wp-block-telex-bento-grid-5 theme-${theme}`,
	});

	const LinkIcon = () => (
		<svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
			<path d="M10 13a5 5 0 0 0 7.54.54l3-3a5 5 0 0 0-7.07-7.07l-1.72 1.71" />
			<path d="M14 11a5 5 0 0 0-7.54-.54l-3 3a5 5 0 0 0 7.07 7.07l1.71-1.71" />
		</svg>
	);

	return (
		<div {...blockProps}>
			<ul className="bento-grid">
				{cards.map((card, index) => {
					const CardContent = () => (
						<>
							{card.image.url && (
								<figure className="card-image">
									<img 
										src={card.image.url} 
										alt={card.image.alt}
										loading="lazy"
									/>
								</figure>
							)}
							<div className="card-content">
								<h3 className="card-title">{card.title}</h3>
								{card.showDescription && card.description && (
									<p className="card-description">{card.description}</p>
								)}
								{card.showIcon && card.link && (
									<div className="card-link-icon">
										<LinkIcon />
									</div>
								)}
							</div>
						</>
					);

					if (card.link) {
						return (
							<li key={card.id} className={`bento-card ${card.customClass ? card.customClass : ''}`}>
								<a href={card.link} className="card-link">
									<CardContent />
								</a>
							</li>
						);
					}

					return (
						<li key={card.id} className={`bento-card ${card.customClass ? card.customClass : ''}`}>
							<CardContent />
						</li>
					);
				})}
			</ul>
		</div>
	);
}
	