
  /**
 * Frontend JavaScript for Bento Grid 5 block
 * Handles interactive behaviors and enhancements
 */

document.addEventListener('DOMContentLoaded', function() {
	// Get all Bento Grid 5 blocks on the page
	const bentoBlocks = document.querySelectorAll('.wp-block-telex-bento-grid-5');
	
	bentoBlocks.forEach(function(block) {
		const cards = block.querySelectorAll('.bento-card');
		
		cards.forEach(function(card) {
			// Add hover effects and accessibility
			card.addEventListener('mouseenter', function() {
				this.style.transform = 'translateY(-4px)';
			});
			
			card.addEventListener('mouseleave', function() {
				this.style.transform = 'translateY(-2px)';
			});
			
			// Handle keyboard navigation for linked cards
			const link = card.querySelector('.card-link');
			if (link) {
				card.setAttribute('tabindex', '0');
				card.setAttribute('role', 'button');
				card.setAttribute('aria-label', 'Navigate to: ' + link.getAttribute('href'));
				
				card.addEventListener('keydown', function(e) {
					if (e.key === 'Enter' || e.key === ' ') {
						e.preventDefault();
						link.click();
					}
				});
			}
			
			// Lazy load images with intersection observer
			const images = card.querySelectorAll('img[loading="lazy"]');
			if ('IntersectionObserver' in window) {
				const imageObserver = new IntersectionObserver(function(entries, observer) {
					entries.forEach(function(entry) {
						if (entry.isIntersecting) {
							const img = entry.target;
							img.classList.add('loaded');
							observer.unobserve(img);
						}
					});
				});
				
				images.forEach(function(img) {
					imageObserver.observe(img);
				});
			}
		});
		
		// Add smooth animations on scroll
		if ('IntersectionObserver' in window) {
			const blockObserver = new IntersectionObserver(function(entries) {
				entries.forEach(function(entry) {
					if (entry.isIntersecting) {
						entry.target.classList.add('animate-in');
						const cards = entry.target.querySelectorAll('.bento-card');
						cards.forEach(function(card, index) {
							setTimeout(function() {
								card.classList.add('fade-in');
							}, index * 100);
						});
					}
				});
			}, {
				threshold: 0.1
			});
			
			blockObserver.observe(block);
		}
	});
	
	// Add CSS animations if not already present
	if (!document.getElementById('bento-grid-animations')) {
		const style = document.createElement('style');
		style.id = 'bento-grid-animations';
		style.textContent = `
			.wp-block-telex-bento-grid-5:not(.animate-in) .bento-card {
				opacity: 0;
				transform: translateY(20px);
			}
			
			.wp-block-telex-bento-grid-5 .bento-card.fade-in {
				opacity: 1;
				transform: translateY(0);
				transition: opacity 0.6s ease, transform 0.6s ease;
			}
			
			.wp-block-telex-bento-grid-5 .card-image img.loaded {
				animation: fadeIn 0.5s ease-in-out;
			}
			
			@keyframes fadeIn {
				from { opacity: 0; }
				to { opacity: 1; }
			}
		`;
		document.head.appendChild(style);
	}
});

// Optional: Add analytics tracking for card clicks
function trackBentoCardClick(cardTitle, cardUrl) {
	// This can be connected to Google Analytics, Adobe Analytics, etc.
	if (typeof gtag !== 'undefined') {
		gtag('event', 'click', {
			'event_category': 'Bento Grid',
			'event_label': cardTitle,
			'value': cardUrl
		});
	}
	
	// Console log for debugging
	console.log('Bento Grid Card Clicked:', { title: cardTitle, url: cardUrl });
}

// Export for potential use in other scripts
window.BentoGrid5 = {
	trackCardClick: trackBentoCardClick
};
	